
/*============================================================================

This C source file is part of TestFloat, Release 3d, a package of programs for
testing the correctness of floating-point arithmetic complying with the IEEE
Standard for Floating-Point, by John R. Hauser.

Copyright 2011, 2012, 2013, 2014 The Regents of the University of California.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice,
    this list of conditions, and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions, and the following disclaimer in the documentation
    and/or other materials provided with the distribution.

 3. Neither the name of the University nor the names of its contributors may
    be used to endorse or promote products derived from this software without
    specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS "AS IS", AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ARE
DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=============================================================================*/

#include <stdbool.h>
#include <stdint.h>
#include "platform.h"
#include "uint128.h"
#include "random.h"
#include "softfloat.h"
#include "genCases.h"

#ifdef FLOAT128

struct sequence {
    int expNum, term1Num, term2Num;
    bool done;
};

enum {
    f128NumQIn  =  22,
    f128NumQOut =  78,
    f128NumP1   =   4,
    f128NumP2   = 443
};
static const uint64_t f128QIn[f128NumQIn] = {
    UINT64_C( 0x0000000000000000 ),    /* positive, subnormal       */
    UINT64_C( 0x0001000000000000 ),    /* positive, -16382          */
    UINT64_C( 0x3F8E000000000000 ),    /* positive,   -113          */
    UINT64_C( 0x3FFD000000000000 ),    /* positive,     -2          */
    UINT64_C( 0x3FFE000000000000 ),    /* positive,     -1          */
    UINT64_C( 0x3FFF000000000000 ),    /* positive,      0          */
    UINT64_C( 0x4000000000000000 ),    /* positive,      1          */
    UINT64_C( 0x4001000000000000 ),    /* positive,      2          */
    UINT64_C( 0x4070000000000000 ),    /* positive,    113          */
    UINT64_C( 0x7FFE000000000000 ),    /* positive,  16383          */
    UINT64_C( 0x7FFF000000000000 ),    /* positive, infinity or NaN */
    UINT64_C( 0x8000000000000000 ),    /* negative, subnormal       */
    UINT64_C( 0x8001000000000000 ),    /* negative, -16382          */
    UINT64_C( 0xBF8E000000000000 ),    /* negative,   -113          */
    UINT64_C( 0xBFFD000000000000 ),    /* negative,     -2          */
    UINT64_C( 0xBFFE000000000000 ),    /* negative,     -1          */
    UINT64_C( 0xBFFF000000000000 ),    /* negative,      0          */
    UINT64_C( 0xC000000000000000 ),    /* negative,      1          */
    UINT64_C( 0xC001000000000000 ),    /* negative,      2          */
    UINT64_C( 0xC070000000000000 ),    /* negative,    113          */
    UINT64_C( 0xFFFE000000000000 ),    /* negative,  16383          */
    UINT64_C( 0xFFFF000000000000 )     /* negative, infinity or NaN */
};
static const uint64_t f128QOut[f128NumQOut] = {
    UINT64_C( 0x0000000000000000 ),    /* positive, subnormal       */
    UINT64_C( 0x0001000000000000 ),    /* positive, -16382          */
    UINT64_C( 0x0002000000000000 ),    /* positive, -16381          */
    UINT64_C( 0x3BFE000000000000 ),    /* positive,  -1025          */
    UINT64_C( 0x3BFF000000000000 ),    /* positive,  -1024          */
    UINT64_C( 0x3C00000000000000 ),    /* positive,  -1023          */
    UINT64_C( 0x3C01000000000000 ),    /* positive,  -1022          */
    UINT64_C( 0x3F7E000000000000 ),    /* positive,   -129          */
    UINT64_C( 0x3F7F000000000000 ),    /* positive,   -128          */
    UINT64_C( 0x3F80000000000000 ),    /* positive,   -127          */
    UINT64_C( 0x3F81000000000000 ),    /* positive,   -126          */
    UINT64_C( 0x3F8E000000000000 ),    /* positive,   -113          */
    UINT64_C( 0x3FFB000000000000 ),    /* positive,     -4          */
    UINT64_C( 0x3FFC000000000000 ),    /* positive,     -3          */
    UINT64_C( 0x3FFD000000000000 ),    /* positive,     -2          */
    UINT64_C( 0x3FFE000000000000 ),    /* positive,     -1          */
    UINT64_C( 0x3FFF000000000000 ),    /* positive,      0          */
    UINT64_C( 0x4000000000000000 ),    /* positive,      1          */
    UINT64_C( 0x4001000000000000 ),    /* positive,      2          */
    UINT64_C( 0x4002000000000000 ),    /* positive,      3          */
    UINT64_C( 0x4003000000000000 ),    /* positive,      4          */
    UINT64_C( 0x401C000000000000 ),    /* positive,     29          */
    UINT64_C( 0x401D000000000000 ),    /* positive,     30          */
    UINT64_C( 0x401E000000000000 ),    /* positive,     31          */
    UINT64_C( 0x401F000000000000 ),    /* positive,     32          */
    UINT64_C( 0x403C000000000000 ),    /* positive,     61          */
    UINT64_C( 0x403D000000000000 ),    /* positive,     62          */
    UINT64_C( 0x403E000000000000 ),    /* positive,     63          */
    UINT64_C( 0x403F000000000000 ),    /* positive,     64          */
    UINT64_C( 0x4070000000000000 ),    /* positive,    113          */
    UINT64_C( 0x407E000000000000 ),    /* positive,    127          */
    UINT64_C( 0x407F000000000000 ),    /* positive,    128          */
    UINT64_C( 0x4080000000000000 ),    /* positive,    129          */
    UINT64_C( 0x43FE000000000000 ),    /* positive,   1023          */
    UINT64_C( 0x43FF000000000000 ),    /* positive,   1024          */
    UINT64_C( 0x4400000000000000 ),    /* positive,   1025          */
    UINT64_C( 0x7FFD000000000000 ),    /* positive,  16382          */
    UINT64_C( 0x7FFE000000000000 ),    /* positive,  16383          */
    UINT64_C( 0x7FFF000000000000 ),    /* positive, infinity or NaN */
    UINT64_C( 0x8000000000000000 ),    /* negative, subnormal       */
    UINT64_C( 0x8001000000000000 ),    /* negative, -16382          */
    UINT64_C( 0x8002000000000000 ),    /* negative, -16381          */
    UINT64_C( 0xBBFE000000000000 ),    /* negative,  -1025          */
    UINT64_C( 0xBBFF000000000000 ),    /* negative,  -1024          */
    UINT64_C( 0xBC00000000000000 ),    /* negative,  -1023          */
    UINT64_C( 0xBC01000000000000 ),    /* negative,  -1022          */
    UINT64_C( 0xBF7E000000000000 ),    /* negative,   -129          */
    UINT64_C( 0xBF7F000000000000 ),    /* negative,   -128          */
    UINT64_C( 0xBF80000000000000 ),    /* negative,   -127          */
    UINT64_C( 0xBF81000000000000 ),    /* negative,   -126          */
    UINT64_C( 0xBF8E000000000000 ),    /* negative,   -113          */
    UINT64_C( 0xBFFB000000000000 ),    /* negative,     -4          */
    UINT64_C( 0xBFFC000000000000 ),    /* negative,     -3          */
    UINT64_C( 0xBFFD000000000000 ),    /* negative,     -2          */
    UINT64_C( 0xBFFE000000000000 ),    /* negative,     -1          */
    UINT64_C( 0xBFFF000000000000 ),    /* negative,      0          */
    UINT64_C( 0xC000000000000000 ),    /* negative,      1          */
    UINT64_C( 0xC001000000000000 ),    /* negative,      2          */
    UINT64_C( 0xC002000000000000 ),    /* negative,      3          */
    UINT64_C( 0xC003000000000000 ),    /* negative,      4          */
    UINT64_C( 0xC01C000000000000 ),    /* negative,     29          */
    UINT64_C( 0xC01D000000000000 ),    /* negative,     30          */
    UINT64_C( 0xC01E000000000000 ),    /* negative,     31          */
    UINT64_C( 0xC01F000000000000 ),    /* negative,     32          */
    UINT64_C( 0xC03C000000000000 ),    /* negative,     61          */
    UINT64_C( 0xC03D000000000000 ),    /* negative,     62          */
    UINT64_C( 0xC03E000000000000 ),    /* negative,     63          */
    UINT64_C( 0xC03F000000000000 ),    /* negative,     64          */
    UINT64_C( 0xC070000000000000 ),    /* negative,    113          */
    UINT64_C( 0xC07E000000000000 ),    /* negative,    127          */
    UINT64_C( 0xC07F000000000000 ),    /* negative,    128          */
    UINT64_C( 0xC080000000000000 ),    /* negative,    129          */
    UINT64_C( 0xC3FE000000000000 ),    /* negative,   1023          */
    UINT64_C( 0xC3FF000000000000 ),    /* negative,   1024          */
    UINT64_C( 0xC400000000000000 ),    /* negative,   1025          */
    UINT64_C( 0xFFFD000000000000 ),    /* negative,  16382          */
    UINT64_C( 0xFFFE000000000000 ),    /* negative,  16383          */
    UINT64_C( 0xFFFF000000000000 )     /* negative, infinity or NaN */
};
static const struct { uint64_t v64, v0; } f128P1[f128NumP1] = {
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000001 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFE ) }
};
static const struct { uint64_t v64, v0; } f128P2[f128NumP2] = {
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000001 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000002 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000004 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000008 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000010 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000020 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000040 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000080 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000100 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000200 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000400 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000800 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000001000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000002000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000004000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000008000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000010000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000020000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000040000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000080000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000100000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000200000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000400000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000800000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000001000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000002000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000004000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000008000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000010000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000020000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000040000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000080000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000100000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000200000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000400000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000800000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000001000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000002000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000004000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000008000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000010000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000020000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000040000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000080000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000100000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000200000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000400000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000800000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0001000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0002000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0004000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0008000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0010000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0020000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0040000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0080000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0100000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0200000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0400000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0800000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x1000000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x2000000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x4000000000000000 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x8000000000000000 ) },
    { UINT64_C( 0x0000000000000001 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000002 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000004 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000008 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000010 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000020 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000040 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000080 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000100 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000200 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000400 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000000800 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000001000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000002000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000004000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000008000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000010000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000020000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000040000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000080000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000100000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000200000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000400000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000000800000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000001000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000002000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000004000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000008000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000010000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000020000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000040000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000080000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000100000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000200000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000400000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000000800000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000001000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000002000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000004000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000008000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000010000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000020000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000040000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000080000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000100000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000200000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000400000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000800000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000C00000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000E00000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000F00000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000F80000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FC0000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FE0000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FF0000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FF8000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFC000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFE000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFF000000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFF800000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFC00000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFE00000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFF00000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFF80000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFC0000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFE0000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFF0000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFF8000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFC000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFE000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFF000000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFF800000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFC00000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFE00000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFF00000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFF80000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFC0000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFE0000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFF0000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFF8000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFC000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFE000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFF000 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFF800 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFC00 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFE00 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFF00 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFF80 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFC0 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFE0 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFF0 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFF8 ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFC ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFE ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0x0000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0x8000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xC000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xE000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xF000000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xF800000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFC00000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFE00000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFF00000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFF80000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFC0000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFE0000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFF0000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFF8000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFC000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFE000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFF000000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFF800000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFC00000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFE00000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFF00000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFF80000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFC0000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFE0000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFF0000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFF8000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFC000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFE000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFF000000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFF800000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFC00000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFE00000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFF00000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFF80000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFC0000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFE0000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFF0000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFF8000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFC000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFE000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFF000000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFF800000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFC00000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFE00000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFF00000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFF80000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFC0000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFE0000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFF0000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFF8000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFC000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFE000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFF000 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFF800 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFC00 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFE00 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFF00 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFF80 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFC0 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFE0 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFF0 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFF8 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFC ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFE ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFD ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFB ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFF7 ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFEF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFDF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFBF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFF7F ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFEFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFDFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFBFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFF7FF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFEFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFDFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFBFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFF7FFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFEFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFDFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFBFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFF7FFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFEFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFDFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFBFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFF7FFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFEFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFDFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFBFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFF7FFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFEFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFDFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFBFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFF7FFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFEFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFDFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFBFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFF7FFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFEFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFDFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFFBFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFF7FFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFEFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFDFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFFBFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFF7FFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFEFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFDFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFFBFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFF7FFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFEFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFDFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFFBFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFF7FFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFEFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFDFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFFBFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFF7FFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFEFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFDFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xFBFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xF7FFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xEFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xDFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0xBFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFF ), UINT64_C( 0x7FFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFD ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFFB ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFF7 ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFEF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFDF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFFBF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFF7F ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFEFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFDFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFFBFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFF7FF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFEFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFDFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFFBFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFF7FFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFEFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFDFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFFBFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFF7FFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFEFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFDFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFFBFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFF7FFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFEFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFDFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFFBFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFF7FFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFEFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFDFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFFBFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFF7FFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFEFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFDFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFFBFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFF7FFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFEFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFDFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FFBFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FF7FFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FEFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FDFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000FBFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000F7FFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000EFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000DFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000BFFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00007FFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00003FFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00001FFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000FFFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000007FFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000003FFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000001FFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000FFFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000007FFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000003FFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000001FFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000FFFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000007FFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000003FFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000001FFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000000FFFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000007FFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000003FFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000001FFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000000FFFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000007FFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000003FFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000001FFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000FFFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000000007FFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000000003FFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000000001FFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000000000FFFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000000007FFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000000003FFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000000001FFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000000000FFFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000007FFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000003FFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000001FFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000FFF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000000000007FF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000000000003FF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000000000001FF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x00000000000000FF ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000000000007F ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000000000003F ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000000000001F ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x000000000000000F ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000007 ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000003 ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000001 ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0xFFFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x7FFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x3FFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x1FFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0FFFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x07FFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x03FFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x01FFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00FFFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x007FFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x003FFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x001FFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000FFFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0007FFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0003FFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0001FFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000FFFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00007FFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00003FFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00001FFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000FFFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000007FFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000003FFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000001FFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000FFFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000007FFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000003FFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000001FFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000FFFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000007FFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000003FFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000001FFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000000FFFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000007FFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000003FFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000001FFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000000FFFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000007FFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000003FFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000001FFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000FFFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000000007FFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000000003FFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000000001FFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000000000FFFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000000007FFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000000003FFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000000001FFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000000000FFFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000007FFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000003FFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000001FFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000FFF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000000000007FF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000000000003FF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000000000001FF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x00000000000000FF ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000000000007F ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000000000003F ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000000000001F ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x000000000000000F ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000007 ) },
    { UINT64_C( 0x0000000000000000 ), UINT64_C( 0x0000000000000003 ) }
};

static const uint_fast64_t f128NumQInP1 = f128NumQIn * f128NumP1;
static const uint_fast64_t f128NumQOutP1 = f128NumQOut * f128NumP1;

static void f128NextQInP1( struct sequence *sequencePtr, float128_t *zPtr )
{
    struct uint128 *uiZPtr;
    int expNum, sigNum;

    uiZPtr = (struct uint128 *) zPtr;
    expNum = sequencePtr->expNum;
    sigNum = sequencePtr->term1Num;
    uiZPtr->v64 = f128QIn[expNum] | f128P1[sigNum].v64;
    uiZPtr->v0  = f128P1[sigNum].v0;
    ++sigNum;
    if ( f128NumP1 <= sigNum ) {
        sigNum = 0;
        ++expNum;
        if ( f128NumQIn <= expNum ) {
            expNum = 0;
            sequencePtr->done = true;
        }
        sequencePtr->expNum = expNum;
    }
    sequencePtr->term1Num = sigNum;

}

static void f128NextQOutP1( struct sequence *sequencePtr, float128_t *zPtr )
{
    struct uint128 *uiZPtr;
    int expNum, sigNum;

    uiZPtr = (struct uint128 *) zPtr;
    expNum = sequencePtr->expNum;
    sigNum = sequencePtr->term1Num;
    uiZPtr->v64 = f128QOut[expNum] | f128P1[sigNum].v64;
    uiZPtr->v0 = f128P1[sigNum].v0;
    ++sigNum;
    if ( f128NumP1 <= sigNum ) {
        sigNum = 0;
        ++expNum;
        if ( f128NumQOut <= expNum ) {
            expNum = 0;
            sequencePtr->done = true;
        }
        sequencePtr->expNum = expNum;
    }
    sequencePtr->term1Num = sigNum;

}

static const uint_fast64_t f128NumQInP2 = f128NumQIn * f128NumP2;
static const uint_fast64_t f128NumQOutP2 = f128NumQOut * f128NumP2;

static void f128NextQInP2( struct sequence *sequencePtr, float128_t *zPtr )
{
    struct uint128 *uiZPtr;
    int expNum, sigNum;

    uiZPtr = (struct uint128 *) zPtr;
    expNum = sequencePtr->expNum;
    sigNum = sequencePtr->term1Num;
    uiZPtr->v64 = f128QIn[expNum] | f128P2[sigNum].v64;
    uiZPtr->v0 = f128P2[sigNum].v0;
    ++sigNum;
    if ( f128NumP2 <= sigNum ) {
        sigNum = 0;
        ++expNum;
        if ( f128NumQIn <= expNum ) {
            expNum = 0;
            sequencePtr->done = true;
        }
        sequencePtr->expNum = expNum;
    }
    sequencePtr->term1Num = sigNum;

}

static void f128NextQOutP2( struct sequence *sequencePtr, float128_t *zPtr )
{
    struct uint128 *uiZPtr;
    int expNum, sigNum;

    uiZPtr = (struct uint128 *) zPtr;
    expNum = sequencePtr->expNum;
    sigNum = sequencePtr->term1Num;
    uiZPtr->v64 = f128QOut[expNum] | f128P2[sigNum].v64;
    uiZPtr->v0 = f128P2[sigNum].v0;
    ++sigNum;
    if ( f128NumP2 <= sigNum ) {
        sigNum = 0;
        ++expNum;
        if ( f128NumQOut <= expNum ) {
            expNum = 0;
            sequencePtr->done = true;
        }
        sequencePtr->expNum = expNum;
    }
    sequencePtr->term1Num = sigNum;

}

static void f128RandomQOutP3( float128_t *zPtr )
{
    struct uint128 *uiZPtr;
    int sigNum1, sigNum2;
    uint_fast64_t sig1_0;

    uiZPtr = (struct uint128 *) zPtr;
    sigNum1 = randomN_ui16( f128NumP2 );
    sigNum2 = randomN_ui16( f128NumP2 );
    sig1_0 = f128P2[sigNum1].v0;
    uiZPtr->v0 = sig1_0 + f128P2[sigNum2].v0;
    uiZPtr->v64 =
          f128QOut[randomN_ui8( f128NumQOut )]
        | ((f128P2[sigNum1].v64 + f128P2[sigNum2].v64 + (uiZPtr->v0 < sig1_0))
               & UINT64_C( 0x0000FFFFFFFFFFFF ));

}

static void f128RandomQOutPInf( float128_t *zPtr )
{
    struct uint128 *uiZPtr;

    uiZPtr = (struct uint128 *) zPtr;
    uiZPtr->v64 =
          f128QOut[randomN_ui8( f128NumQOut )]
        | (random_ui64() & UINT64_C( 0x0000FFFFFFFFFFFF ));
    uiZPtr->v0 = random_ui64();

}

enum { f128NumQInfWeightMasks = 14 };
static const uint64_t f128QInfWeightMasks[f128NumQInfWeightMasks] = {
    UINT64_C( 0xFFFF000000000000 ),
    UINT64_C( 0xFFFF000000000000 ),
    UINT64_C( 0xBFFF000000000000 ),
    UINT64_C( 0x9FFF000000000000 ),
    UINT64_C( 0x87FF000000000000 ),
    UINT64_C( 0x87FF000000000000 ),
    UINT64_C( 0x83FF000000000000 ),
    UINT64_C( 0x81FF000000000000 ),
    UINT64_C( 0x80FF000000000000 ),
    UINT64_C( 0x807F000000000000 ),
    UINT64_C( 0x803F000000000000 ),
    UINT64_C( 0x801F000000000000 ),
    UINT64_C( 0x800F000000000000 ),
    UINT64_C( 0x8007000000000000 )
};
static const uint64_t f128QInfWeightOffsets[f128NumQInfWeightMasks] = {
    UINT64_C( 0x0000000000000000 ),
    UINT64_C( 0x0000000000000000 ),
    UINT64_C( 0x2000000000000000 ),
    UINT64_C( 0x3000000000000000 ),
    UINT64_C( 0x3800000000000000 ),
    UINT64_C( 0x3C00000000000000 ),
    UINT64_C( 0x3E00000000000000 ),
    UINT64_C( 0x3F00000000000000 ),
    UINT64_C( 0x3F80000000000000 ),
    UINT64_C( 0x3FC0000000000000 ),
    UINT64_C( 0x3FE0000000000000 ),
    UINT64_C( 0x3FF0000000000000 ),
    UINT64_C( 0x3FF8000000000000 ),
    UINT64_C( 0x3FFC000000000000 )
};

static void f128RandomQInfP3( float128_t *zPtr )
{
    struct uint128 *uiZPtr;
    int sigNum1, sigNum2;
    uint_fast64_t sig1_0;
    int weightMaskNum;

    uiZPtr = (struct uint128 *) zPtr;
    sigNum1 = randomN_ui16( f128NumP2 );
    sigNum2 = randomN_ui16( f128NumP2 );
    sig1_0 = f128P2[sigNum1].v0;
    uiZPtr->v0 = sig1_0 + f128P2[sigNum2].v0;
    weightMaskNum = randomN_ui8( f128NumQInfWeightMasks );
    uiZPtr->v64 =
          (((uint_fast64_t) random_ui16()<<48
                & f128QInfWeightMasks[weightMaskNum])
               + f128QInfWeightOffsets[weightMaskNum])
        | ((f128P2[sigNum1].v64 + f128P2[sigNum2].v64 + (uiZPtr->v0 < sig1_0))
               & UINT64_C( 0x0000FFFFFFFFFFFF ));

}

static void f128RandomQInfPInf( float128_t *zPtr )
{
    struct uint128 *uiZPtr;
    int weightMaskNum;

    uiZPtr = (struct uint128 *) zPtr;
    weightMaskNum = randomN_ui8( f128NumQInfWeightMasks );
    uiZPtr->v64 =
        (random_ui64()
             & (f128QInfWeightMasks[weightMaskNum]
                    | UINT64_C( 0x0000FFFFFFFFFFFF )))
            + f128QInfWeightOffsets[weightMaskNum];
    uiZPtr->v0 = random_ui64();

}

static void f128Random( float128_t *zPtr )
{

    switch ( random_ui8() & 7 ) {
     case 0:
     case 1:
     case 2:
        f128RandomQOutP3( zPtr );
        break;
     case 3:
        f128RandomQOutPInf( zPtr );
        break;
     case 4:
     case 5:
     case 6:
        f128RandomQInfP3( zPtr );
        break;
     case 7:
        f128RandomQInfPInf( zPtr );
        break;
    }

}

static struct sequence sequenceA, sequenceB, sequenceC;
static float128_t currentA, currentB, currentC;
static int subcase;

float128_t genCases_f128_a, genCases_f128_b, genCases_f128_c;

void genCases_f128_a_init( void )
{

    sequenceA.expNum = 0;
    sequenceA.term1Num = 0;
    sequenceA.term2Num = 0;
    sequenceA.done = false;
    subcase = 0;
    genCases_total =
        (genCases_level == 1) ? 3 * f128NumQOutP1 : 2 * f128NumQOutP2;
    genCases_done = false;

}

void genCases_f128_a_next( void )
{

    if ( genCases_level == 1 ) {
        switch ( subcase ) {
         case 0:
         case 1:
            f128Random( &genCases_f128_a );
            break;
         case 2:
            f128NextQOutP1( &sequenceA, &genCases_f128_a );
            genCases_done = sequenceA.done;
            subcase = -1;
            break;
        }
     } else {
        switch ( subcase ) {
         case 0:
            f128Random( &genCases_f128_a );
            break;
         case 1:
            f128NextQOutP2( &sequenceA, &genCases_f128_a );
            genCases_done = sequenceA.done;
            subcase = -1;
            break;
        }
    }
    ++subcase;

}

void genCases_f128_ab_init( void )
{

    sequenceA.expNum = 0;
    sequenceA.term1Num = 0;
    sequenceA.term2Num = 0;
    sequenceA.done = false;
    sequenceB.expNum = 0;
    sequenceB.term1Num = 0;
    sequenceB.term2Num = 0;
    sequenceB.done = false;
    subcase = 0;
    if ( genCases_level == 1 ) {
        genCases_total = 6 * f128NumQInP1 * f128NumQInP1;
        f128NextQInP1( &sequenceA, &currentA );
    } else {
        genCases_total = 2 * f128NumQInP2 * f128NumQInP2;
        f128NextQInP2( &sequenceA, &currentA );
    }
    genCases_done = false;

}

void genCases_f128_ab_next( void )
{

    if ( genCases_level == 1 ) {
        switch ( subcase ) {
         case 0:
            if ( sequenceB.done ) {
                sequenceB.done = false;
                f128NextQInP1( &sequenceA, &currentA );
            }
            f128NextQInP1( &sequenceB, &currentB );
         case 2:
         case 4:
            f128Random( &genCases_f128_a );
            f128Random( &genCases_f128_b );
            break;
         case 1:
            genCases_f128_a = currentA;
            f128Random( &genCases_f128_b );
            break;
         case 3:
            f128Random( &genCases_f128_a );
            genCases_f128_b = currentB;
            break;
         case 5:
            genCases_f128_a = currentA;
            genCases_f128_b = currentB;
            genCases_done = sequenceA.done & sequenceB.done;
            subcase = -1;
            break;
        }
    } else {
        switch ( subcase ) {
         case 0:
            f128Random( &genCases_f128_a );
            f128Random( &genCases_f128_b );
            break;
         case 1:
            if ( sequenceB.done ) {
                sequenceB.done = false;
                f128NextQInP2( &sequenceA, &currentA );
            }
            genCases_f128_a = currentA;
            f128NextQInP2( &sequenceB, &genCases_f128_b );
            genCases_done = sequenceA.done & sequenceB.done;
            subcase = -1;
            break;
        }
    }
    ++subcase;

}

void genCases_f128_abc_init( void )
{

    sequenceA.expNum = 0;
    sequenceA.term1Num = 0;
    sequenceA.term2Num = 0;
    sequenceA.done = false;
    sequenceB.expNum = 0;
    sequenceB.term1Num = 0;
    sequenceB.term2Num = 0;
    sequenceB.done = false;
    sequenceC.expNum = 0;
    sequenceC.term1Num = 0;
    sequenceC.term2Num = 0;
    sequenceC.done = false;
    subcase = 0;
    if ( genCases_level == 1 ) {
        genCases_total = 9 * f128NumQInP1 * f128NumQInP1 * f128NumQInP1;
        f128NextQInP1( &sequenceA, &currentA );
        f128NextQInP1( &sequenceB, &currentB );
    } else {
        genCases_total = 2 * f128NumQInP2 * f128NumQInP2 * f128NumQInP2;
        f128NextQInP2( &sequenceA, &currentA );
        f128NextQInP2( &sequenceB, &currentB );
    }
    genCases_done = false;

}

void genCases_f128_abc_next( void )
{

    if ( genCases_level == 1 ) {
        switch ( subcase ) {
         case 0:
            if ( sequenceC.done ) {
                sequenceC.done = false;
                if ( sequenceB.done ) {
                    sequenceB.done = false;
                    f128NextQInP1( &sequenceA, &currentA );
                }
                f128NextQInP1( &sequenceB, &currentB );
            }
            f128NextQInP1( &sequenceC, &currentC );
            f128Random( &genCases_f128_a );
            f128Random( &genCases_f128_b );
            genCases_f128_c = currentC;
            break;
         case 1:
            genCases_f128_a = currentA;
            genCases_f128_b = currentB;
            f128Random( &genCases_f128_c );
            break;
         case 2:
            f128Random( &genCases_f128_a );
            f128Random( &genCases_f128_b );
            f128Random( &genCases_f128_c );
            break;
         case 3:
            f128Random( &genCases_f128_a );
            genCases_f128_b = currentB;
            genCases_f128_c = currentC;
            break;
         case 4:
            genCases_f128_a = currentA;
            f128Random( &genCases_f128_b );
            f128Random( &genCases_f128_c );
            break;
         case 5:
            f128Random( &genCases_f128_a );
            genCases_f128_b = currentB;
            f128Random( &genCases_f128_c );
            break;
         case 6:
            genCases_f128_a = currentA;
            f128Random( &genCases_f128_b );
            genCases_f128_c = currentC;
            break;
         case 7:
            f128Random( &genCases_f128_a );
            f128Random( &genCases_f128_b );
            f128Random( &genCases_f128_c );
            break;
         case 8:
            genCases_f128_a = currentA;
            genCases_f128_b = currentB;
            genCases_f128_c = currentC;
            genCases_done = sequenceA.done & sequenceB.done & sequenceC.done;
            subcase = -1;
            break;
        }
    } else {
        switch ( subcase ) {
         case 0:
            f128Random( &genCases_f128_a );
            f128Random( &genCases_f128_b );
            f128Random( &genCases_f128_c );
            break;
         case 1:
            if ( sequenceC.done ) {
                sequenceC.done = false;
                if ( sequenceB.done ) {
                    sequenceB.done = false;
                    f128NextQInP2( &sequenceA, &currentA );
                }
                f128NextQInP2( &sequenceB, &currentB );
            }
            genCases_f128_a = currentA;
            genCases_f128_b = currentB;
            f128NextQInP2( &sequenceC, &genCases_f128_c );
            genCases_done = sequenceA.done & sequenceB.done & sequenceC.done;
            subcase = -1;
            break;
        }
    }
    ++subcase;

}

#endif

